Imports System.Xml
Imports System.Xml.Xsl
Imports DotNetNuke
Imports DotNetNuke.Services.Exceptions
Imports System.Net.Mail

Namespace Business
 Public Class CheckReports
  Inherits DotNetNuke.Services.Scheduling.SchedulerClient

  Public Sub New(ByVal objScheduleHistoryItem As DotNetNuke.Services.Scheduling.ScheduleHistoryItem)
   MyBase.new()
   Me.ScheduleHistoryItem = objScheduleHistoryItem
  End Sub

  Public Overrides Sub DoWork()

   Try

    CheckReports()
    Me.ScheduleHistoryItem.Succeeded = True

   Catch ex As Exception

    Me.ScheduleHistoryItem.Succeeded = False
    Me.ScheduleHistoryItem.AddLogNote("Generating reports failed: " & ex.Message & "(" & ex.StackTrace & ")<br />")
    Me.Errored(ex)
    LogException(ex)

   End Try

  End Sub

  Public Sub CheckReports()

   Dim NrReportsSent As Integer = 0

   Dim dmc As New DotNetNuke.Entities.Modules.DesktopModuleController
   Dim dm As DotNetNuke.Entities.Modules.DesktopModuleInfo = dmc.GetDesktopModuleByModuleName("Bring2mind\SiteReport")
   If dm Is Nothing Then
    ScheduleHistoryItem.AddLogNote("No Site Report Installed")
    Exit Sub
   End If

   Dim mdc As New DotNetNuke.Entities.Modules.Definitions.ModuleDefinitionController
   Dim md As DotNetNuke.Entities.Modules.Definitions.ModuleDefinitionInfo = mdc.GetModuleDefinitionByName(dm.DesktopModuleID, "Site Report")
   If md Is Nothing Then
    ScheduleHistoryItem.AddLogNote("No Site Report Installed")
    Exit Sub
   End If

   Dim mc As New Entities.Modules.ModuleController
   Dim m As Entities.Modules.ModuleInfo

   Dim pc As New Entities.Portals.PortalController
   Dim pi As Entities.Portals.PortalInfo

   For Each pi In pc.GetPortals
    For Each m In mc.GetModulesByDefinition(pi.PortalID, "Site Report")
     If m.ModuleDefID = md.ModuleDefID Then
      ScheduleHistoryItem.AddLogNote("Found Module " & m.ModuleID.ToString & "<br />")
      Dim Settings As New ModuleSettings(m.ModuleID)
      If (Settings.NextSend < Now) Then

       ' get the timing
       ScheduleHistoryItem.AddLogNote("Have to send report because it's past " & Settings.NextSend.ToUniversalTime.ToString("u") & "<br />")
       Dim EndTime As Date = Settings.NextSend
       Dim StartTime As Date
       Select Case Settings.Interval
        Case Interval.Daily
         StartTime = EndTime.AddHours(-24)
        Case Interval.Weekly
         StartTime = EndTime.AddDays(-7)
        Case Interval.Monthly
         StartTime = EndTime.AddMonths(-1)
       End Select

       ' prepare path to save reports
       pi = pc.GetPortal(m.PortalID)
       Dim tmpPath As String = IO.Path.Combine(IO.Path.Combine(DotNetNuke.Common.ApplicationMapPath, pi.HomeDirectory.Replace("/", "\")), "Reports\")
       If Not IO.Directory.Exists(tmpPath) Then
        IO.Directory.CreateDirectory(tmpPath)
       End If

       ' prepare the transformer
       Dim trans As New XslCompiledTransform
       Dim sw As New IO.StringWriter
       Dim xslf As String = "DesktopModules\Bring2mind\SiteReport\MailTrans.xsl"
       If Settings.MailStyleSheet <> "" Then
        xslf = pi.HomeDirectory.Replace("/", "\") & "\"
        xslf &= Settings.MailStyleSheet
       End If
       trans.Load(DotNetNuke.Common.ApplicationMapPath & "\" & xslf)

       ' make reports
       Dim reportFiles As New List(Of String)
       ScheduleHistoryItem.AddLogNote("Creating reports from " & StartTime.ToUniversalTime.ToString("u") & " to " & EndTime.ToString("u") & "<br />")
       For Each moduleReport As ReportInfo In ReportsController.GetReportsByModule(m.ModuleID)
        Dim rep As New Report(m, moduleReport, StartTime, EndTime)
        ScheduleHistoryItem.AddLogNote("Created report " & moduleReport.Title & "<br />")
        ' Save the report
        Dim tmpFile As String = tmpPath & moduleReport.Title & " " & Format(Now, "yyyy-MM-dd")
        Dim i As Integer = 0
        Do While IO.File.Exists(tmpFile & i.ToString & ".xml")
         i += 1
        Loop
        tmpFile = tmpFile & i.ToString & ".xml"
        rep.Report.Save(tmpFile)
        ScheduleHistoryItem.AddLogNote("Saved report<br />")
        Dim Xml As XmlReader = New XmlNodeReader(rep.Report)
        If Not trans Is Nothing And Not Xml Is Nothing Then
         trans.Transform(Xml, Nothing, sw)
        End If
        reportFiles.Add(tmpFile)
       Next

       ' Now send the email and delete the temporary file
       Dim body As String = Regex.Replace(sw.ToString, "<\?.*\?>\r\n", "")
       Dim message As New MailMessage()
       With message
        .From = New MailAddress(pi.Email, pi.PortalName)
        .To.Add(New MailAddress(pi.Email, pi.PortalName))
        .SubjectEncoding = System.Text.Encoding.UTF8
        .Subject = m.ModuleTitle & " (" & Settings.NextSend.ToShortDateString & " " & Settings.NextSend.ToShortTimeString & ")"
        .BodyEncoding = System.Text.Encoding.UTF8
        .IsBodyHtml = True
        .Body = body
       End With

       ' Make the distribution list
       Dim lst As New StringBuilder
       Using ir As IDataReader = Data.DataProvider.Instance.GetUsersByModulePermission(m.ModuleID, EmailPermissionKey)
        Do While ir.Read
         If Settings.SendAdmins OrElse (GetIntegerValue(ir.Item("RoleId")) <> pi.AdministratorRoleId) Then
          message.Bcc.Add(New MailAddress(CStr(ir.Item("Email"))))
          lst.Append(CStr(ir.Item("Email")))
          lst.Append(";")
         End If
        Loop
       End Using
       If Settings.EmailSQL.Trim <> "" Then
        Using ir As IDataReader = DotNetNuke.Data.DataProvider.Instance.ExecuteSQL(Settings.EmailSQL)
         If ir Is Nothing Then
          ScheduleHistoryItem.AddLogNote("SQL statement for email distribution list contains errors!<br />")
         Else
          Try
           Do While ir.Read
            message.Bcc.Add(New MailAddress(CStr(ir.Item("Email"))))
            lst.Append(CStr(ir.Item("Email")))
            lst.Append(";")
           Loop
          Catch ex As Exception
           ScheduleHistoryItem.AddLogNote(String.Format("SQL statement for email distribution list did not return expected result: {0}<br />", ex.Message))
          End Try
         End If
        End Using
       End If

       ' Add files
       For Each reportFile As String In reportFiles
        Dim att As New Net.Mail.Attachment(reportFile)
        att.Name = IO.Path.GetFileName(reportFile)
        message.Attachments.Add(att)
       Next

       ' send email
       ScheduleHistoryItem.AddLogNote("Sending email to " & lst.ToString & "<br />")
       Dim res As String = SendMail(message)
       If res <> "" Then
        ScheduleHistoryItem.AddLogNote(SendMail(message) & "<br />")
       End If

       ' Delete files
       For Each reportFile As String In reportFiles
        Try
         IO.File.Delete(reportFile)
         ScheduleHistoryItem.AddLogNote("Deleted file.<br />")
        Catch ex As Exception
         ScheduleHistoryItem.AddLogNote("Could not delete file " & reportFile & "<br />")
        End Try
       Next

       ' Set the next time to send
       Select Case Settings.Interval
        Case Interval.Daily
         Settings.NextSend = EndTime.AddHours(24)
        Case Interval.Weekly
         Settings.NextSend = EndTime.AddDays(7)
        Case Interval.Monthly
         Settings.NextSend = EndTime.AddMonths(1)
       End Select
       ScheduleHistoryItem.AddLogNote("Next send is " & Settings.NextSend.ToUniversalTime.ToString("u") & "<br />")
       Settings.SaveSettings(m.ModuleID)
       ScheduleHistoryItem.AddLogNote("Saved settings for module " & m.ModuleID.ToString & "<br />")

       NrReportsSent += 1

      End If
     End If
    Next
   Next

   ScheduleHistoryItem.AddLogNote("Sent " & NrReportsSent.ToString & " reports<br />")

  End Sub

 End Class
End Namespace